#[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
# Import the logging function
. ./logging.ps1
. ./functions.ps1
# Read the configuration file
$configFilePath = "config.json"
$config = Get-Content -Path $configFilePath | ConvertFrom-Json

# Verify that the configuration parameters are set
Check-Parameters -configFilePath $configFilePath

# Construct the token URL
$tokenUrl = $config.webapi_base_url + "/tse_webapi/auth/token"
$token_api = "Bearer " + $config.token
Log-Output "Token URL: $tokenUrl" $config.log_to_file

# Set headers for the token request
$headers = New-Object "System.Collections.Generic.Dictionary[[String],[String]]"
$headers.Add("Content-Type", "application/x-www-form-urlencoded")
$headers.Add("Access-Control-Allow-Origin", "http://localhost:3000")

# Set body for the token request
$body = New-Object "System.Collections.Generic.Dictionary[[String],[String]]"
$body.Add("username", $config.username)
$body.Add("token", $token_api) 
$body.Add("scope", $config.scope)
$body.Add("grant_type", $config.grant_type)

# Request the access token
try {
    $tokenResponse = Invoke-RestMethod -Method 'POST' -Uri $tokenUrl -Body $body -Headers $headers
    $accessToken = $tokenResponse.access_token
    Log-Output "Token di accesso ricevuto!" $config.log_to_file
} catch {
    Log-Output "Errore durante l'ottenimento del token: $_" $config.log_to_file
    exit
}

# Get the list of files to upload
$filesToUpload = Get-ChildItem -Path $config.upload_folder | Where-Object { $_.Name -match $config.file_extension_regex }

# Log the number of files to be uploaded
$filesCount = $filesToUpload.Count
Log-Output "Numero di file da caricare: $filesCount" $config.log_to_file


# get list files from server and save in files.json 
Get-Files -fixedPathType $config.fixed_path_type -relativePath $config.download_relative_path

# Read the list of files to download from files.json
$filesConfig = Get-Content -Path "files.json" | ConvertFrom-Json
$filesToDownload = $filesConfig.files
# Log the number of files to be downloaded
$filesDownloadCount = $filesToDownload.Count
Log-Output "Numero di file da scaricare: $filesDownloadCount" $config.log_to_file

# Initialize the count of successfully downloaded files
$successfulDownloads = 0


# Download each file listed in files.json
foreach ($fileName in $filesToDownload) {
    $destinationPath = Join-Path -Path $config.download_folder -ChildPath $fileName
    
    $result = Download-File -fileName $fileName -destinationPath $destinationPath

    # Increment the count if the download was successful
    if ($result -eq $true) {
        $successfulDownloads++
    }

    # Remove the file if delete_after_download is true and the download was successful
    if ($config.delete_after_download -and $result -eq $true) {
        Remove-File -fileName $fileName
    }
}

# Log the total number of files downloaded
Log-Output "Numero totale di file scaricati con successo: $successfulDownloads" $config.log_to_file

# Upload each file
foreach ($file in $filesToUpload) {
    $filePath = $file.FullName
    Log-Output("Upload file: $filePath") $config.log_to_file
    Upload-File -filePath $filePath
}



