# Function to authenticate and get the access token
function Authenticate {
    param (
        [string]$configFilePath
    )
    $config = Get-Content -Path $configFilePath | ConvertFrom-Json
    # Construct the token URL
    $tokenUrl = $config.webapi_base_url + "/tse_webapi/auth/token"
    $token_api = "Bearer " + $config.token
    Log-Output "Token URL: $tokenUrl" $config.log_to_file

    # Set headers for the token request
    $headers = New-Object "System.Collections.Generic.Dictionary[[String],[String]]"
    $headers.Add("Content-Type", "application/x-www-form-urlencoded")
    $headers.Add("Access-Control-Allow-Origin", "http://localhost:3000")

    # Set body for the token request
    $body = New-Object "System.Collections.Generic.Dictionary[[String],[String]]"
    $body.Add("username", $config.username)
    $body.Add("token", $token_api) 
    $body.Add("scope", $config.scope)
    $body.Add("grant_type", $config.grant_type)

    # Request the access token
    try {
        $tokenResponse = Invoke-RestMethod -Method 'POST' -Uri $tokenUrl -Body $body -Headers $headers
        $accessToken = $tokenResponse.access_token
        Log-Output "Token di accesso ricevuto!" $config.log_to_file
        return $accessToken
    } catch {
        Log-Output "Errore durante l'ottenimento del token: $_" $config.log_to_file
        exit
    }
}

# Function to download a file
function Download-File {
    param (
        [string]$fileName,
        [string]$destinationPath
    )
    $downloadUrl = $config.webapi_base_url + "/tse_webapi/v1/" + $config.scope + "/FW/EditPathFW/getfile"
    $headers = @{
        "Authorization-Scope" = $config.scope
        "Authorization" = "Bearer $accessToken"
        "Content-Type" = "application/json"
    }
    $body = @{
        asAttachment = $true
        deleteAfter = $false
        filename = $fileName
        fixedPathType = $config.fixed_path_type
        relativePath = $config.download_relative_path
    }
    try {
        log-output "Download file: $fileName" $config.log_to_file
        $response = Invoke-RestMethod -Method Post -Uri $downloadUrl -Headers $headers -Body ($body | ConvertTo-Json) -OutFile $destinationPath
        Log-Output "File scaricato con successo: $destinationPath" $config.log_to_file
        #Log-Output "HTTP Status Code: $($response.StatusCode)" $config.log_to_file
        return $true
    } catch {
        Log-Output "Errore durante il download del file: $_" $config.log_to_file
        return $false
    }
}

# Function to get the list of files
function Get-Files {
    param (
        [string]$fixedPathType,
        [string]$relativePath
    )
    $downloadUrl = $config.webapi_base_url + "/tse_webapi/v1/" + $config.scope + "/FW/EditPathFW/getfilelist"
    $headers = @{
        "Authorization-Scope" = $config.scope
        "Authorization" = "Bearer $accessToken"
        "Content-Type" = "application/json"
    }
    $body = @{
        fixedPathType = $fixedPathType
        relativePath = $relativePath
    }
    try {
        log-output "Get files: $fileName" $config.log_to_file
        $response = Invoke-RestMethod -Method Post -Uri $downloadUrl -Headers $headers -Body ($body | ConvertTo-Json) -OutFile "files.json"
        Log-Output "Lista file scaricata in files.json" $config.log_to_file
        #Log-Output "HTTP Status Code: $($response.StatusCode)" $config.log_to_file
        #return $true
    } catch {
        Log-Output "Errore durante il download della lista: $_" $config.log_to_file
        #return $false
    }
}

# Function to remove a file
function Remove-File {
    param (
        [string]$fileName
    )
    $removeUrl = $config.webapi_base_url + "/tse_webapi/v1/" + $config.scope + "/FW/EditPathFW/removefile"
    $headers = @{
        "Authorization-Scope" = $config.scope
        "Authorization" = "Bearer $accessToken"
        "Content-Type" = "application/json"
    }

    $body = @{
        filename = $fileName
        fixedPathType = $config.fixed_path_type
        relativePath = $config.download_relative_path
    }

    try {
        log-output "Remove file: $fileName" $config.log_to_file
        $response = Invoke-RestMethod -Method Post -Uri   $removeUrl -Headers $headers -Body ($body | ConvertTo-Json)
        Log-Output "File rimosso con successo: $fileName" $config.log_to_file
    } catch {
        Log-Output "Errore durante la rimozione del file: $_" $config.log_to_file
    }
}

# Function to upload a file
function Upload-File {
    param (
        [string]$filePath
    )
    $uploadUrl = $config.webapi_base_url + "/tse_webapi/v1/" + $config.scope + "/FW/EditPathFW/uploadfile"
    $multipartContent = @{
        fixedPathType = $config.fixed_path_type
        relativePath = $config.upload_relative_path
        file = Get-Item -Path $filePath
    }
    try {
        $response = Invoke-RestMethod -Method Post -Uri $uploadUrl -Headers @{
            "Authorization-Scope" = $config.scope
            "Authorization" = "Bearer $accessToken"
        } -Form $multipartContent -StatusCodeVariable statusCode
        Log-Output "File: $filePath , caricato con successo!" $config.log_to_file
        #Log-Output "HTTP Status Code: $statusCode" $config.log_to_file
        
        # Move the file to the backup folder if specified
        if ($config.backup_folder) {
            $timestamp = Get-Date -Format "yyyyMMddHHmmss"
            $backupFileName = [System.IO.Path]::GetFileNameWithoutExtension($filePath) + "_$timestamp" + [System.IO.Path]::GetExtension($filePath)
            $backupPath = Join-Path -Path $config.backup_folder -ChildPath $backupFileName
            Move-Item -Path $filePath -Destination $backupPath
            Log-Output "File spostato nella cartella di backup: $backupPath" $config.log_to_file
        }
    } catch {
        Log-Output "Errore durante la lettura del file: $_" $config.log_to_file
        exit
    }
}

# Function to check configuration parameters
function Check-Parameters {
    param (
        [string]$configFilePath
    )
    $config = Get-Content -Path $configFilePath | ConvertFrom-Json
    if (-not $config.webapi_base_url) {
        Log-Output "Error: 'webapi_base_url' is not set in the configuration file." $config.log_to_file
        exit
    }
    if (-not $config.username) {
        Log-Output "Error: 'username' is not set in the configuration file." $config.log_to_file
        exit
    }
    if (-not $config.token) {
        Log-Output "Error: 'token' is not set in the configuration file." $config.log_to_file
        exit
    }
    if (-not $config.scope) {
        Log-Output "Error: 'scope' is not set in the configuration file." $config.log_to_file
        exit
    }
    if (-not $config.grant_type) {
        Log-Output "Error: 'grant_type' is not set in the configuration file." $config.log_to_file
        exit
    }
    if (-not $config.log_to_file) {
        $config.log_to_file = $false
    }
    if (-not $config.backup_folder) {
        Log-Output "Alert: 'backup_folder' is not set in the configuration file. No backup will be made." $config.log_to_file
    } else {
        # Check if the backup folder exists, if not, create it
        if (-not (Test-Path -Path $config.backup_folder)) {
            New-Item -ItemType Directory -Path $config.backup_folder | Out-Null
            Log-Output "Cartella di backup creata: $config.backup_folder" $config.log_to_file
        }
    }
    if (-not $config.upload_folder) {
        Log-Output "Error: 'upload_folder' is not set in the configuration file." $config.log_to_file
        exit
    }
    if (-not $config.download_folder) {
        Log-Output "Error: 'download_folder' is not set in the configuration file." $config.log_to_file
        exit
    }
    if (-not $config.file_extension_regex) {
        Log-Output "Error: 'file_extension_regex' is not set in the configuration file." $config.log_to_file
        exit
    }
}